import {
    addCustomDomainSupport,
    addDomain
} from './lib/extension_script.js';
import {
    MessageKeys,
    appCookiesName,
    getTicketingSystemType,
    isCurrentDomainAlreadyAdded
} from './lib/common.js';


let appDomain = "";
let authWindowId = null;
let authParentWindowTabId = null;


const PIA_AUTH_TEST_PATH = "/Extension/VerifyAuthorization";
const PIA_API_EXTENSION_AUTH_PATH = "/Extension/Authenticate";
chrome.runtime.onInstalled.addListener(() => {

});

const verifyAllCookies = function (callback) {
    chrome.cookies.getAll({}, function (cookies) {
        callback(cookies);
    });
}

/**
 * Ensure PiaTabInfo chrome sync storage item.
 * Contains a side effect - updates the `appDomain` global variable
 * @param {string} ticketNumber
 * @param {(setting: { ticketNumber: string, autoLunch: boolean, cookieExists: boolean, serverUrl: string }) => Promise<void>} callback
 * @returns {Promise<void>}
 */
const ensurePiaTabInfo = async function (ticketNumber, callback) {
    let setting = { ticketNumber: ticketNumber, autoLunch: false, cookieExists: false, serverUrl: "" };
    let data = await chrome.storage.sync.get(['PiaTabInfo']);

    if (typeof (data) === "undefined") {
        // ensure that the PiaTabInfo item exists in chrome sync storage
        await chrome.storage.sync.set({ PiaTabInfo: {} });
    }
    else {
        let currentTabInfos = data.PiaTabInfo;

        if (typeof (currentTabInfos) !== "undefined") {

            if (currentTabInfos.serverUrl) {
                setting.serverUrl = currentTabInfos.serverUrl;
                // side effect-  update AppDomain
                appDomain = currentTabInfos.serverUrl;
            }

            setting.autoLunch = !!currentTabInfos.autoLunch;
        }
    }

    if (typeof (callback) === "function") {
        await callback(setting);
    }
}

const sendMessageToContentScript = async function (tabid, messageType, data) {
    let message = { messageType: messageType, data: data };
    await chrome.tabs.sendMessage(tabid, { message: message });
}

const getDomainNameFromUrl = function () {
    if (typeof(appDomain) === "undefined" || appDomain.trim() == "") {
        return "";
    }
    let domain = getDomainName(appDomain);
    return domain;
}

const getDomainName = function (url) {
    let domain = (new URL(url));
    domain = domain.hostname;
    return domain;
}

const createAuthWindow = async function (parentTabId, serveraddress) {
    if (authWindowId != null) {
        chrome.windows.update(authWindowId, { focused: true });
        await sendMessageToContentScript(parentTabId, MessageKeys.PIA_CREATE_AUTH_WINDOW_CREATED, { windowId: authWindowId });
    }
    else {
        let serverAuthAddress = serveraddress + PIA_API_EXTENSION_AUTH_PATH;
        chrome.windows.create({
            url: serverAuthAddress,
            type: "popup"
        }, async function (windowInfo) {
            authWindowId = windowInfo.id;
            authParentWindowTabId = parentTabId;
            await sendMessageToContentScript(parentTabId, MessageKeys.PIA_CREATE_AUTH_WINDOW_CREATED, { windowId: authWindowId });
        });
    }
}

const notifyLoginSessionInfo = async function (tabid, serverAddress) {
    verifyAllCookies(async function (cookies) {
        let domain = getDomainNameFromUrl();
        if (serverAddress != null) {
            domain = getDomainName(serverAddress);
        }

        let appCookies = cookies.filter(function (cookie) {
            return (cookie.domain == domain && appCookiesName.filter(f => f == cookie.name).length > 0);
        });
        let cookieFound = false;
        if (appCookies.length > 0) {
            cookieFound = true;
        }
        if (cookieFound && authWindowId != null) {
            closeTab(authWindowId, function () {
                authWindowId = null;
                authParentWindowTabId = null;
            });
        }
        await sendMessageToContentScript(tabid, MessageKeys.PIA_COOKIE_INFO_QUERY_REPLY, { cookieExists: cookieFound });
    });
}

const closeTab = function (tabId, callback) {
    chrome.windows.remove(tabId, callback);
}


const isClientAuthorized = async function (senderId, authEndPointAddress, browserURL, callback) {
    let authorized = false;
    try {
        let currentTicketingSystem = getTicketingSystemType(browserURL);
        if (currentTicketingSystem == null) {
            //get the ticketing system for custom domain
            let existingCustomDomainInfo = await isCurrentDomainAlreadyAdded(browserURL);
            if (!existingCustomDomainInfo) {
                currentTicketingSystem = "";
            }
            else {
                currentTicketingSystem = existingCustomDomainInfo.system;
            }
        }
        let fullUrl = authEndPointAddress + PIA_AUTH_TEST_PATH + "?ticketingSystem=" + currentTicketingSystem;
        let response = await fetch(fullUrl);
        if (response.status === 200) {
            if (response.redirected !== true) {
                let data = await response.json();
                authorized = data.success;
            }
        }
    }
    catch {
        // The try catch is here to prevent extension from failing further processing just in case there were some errors on the server side.
    }

    if (callback != null) {
        callback(senderId, authorized);
    }
}

// Any conditions that call the async method, need to return true. This will indicate that the calling section that the current operation is
// async and prevents the error Uncaught (in Promise) Error. So, the implementation does not await any call and immediately returns true at the end.
// https://developer.chrome.com/docs/extensions/develop/concepts/messaging
chrome.runtime.onMessage.addListener(async (request, sender, sendResponse) => {
    sendResponse({ status: true });

    if (request.messageType == MessageKeys.PIA_ADD_CUSTOM_TICKETING_SYSTEM) {//save required ticketingsystem info
        initiatePiaFromPopup(request.data);
    }
    else if (request.messageType == MessageKeys.PIA_INITIATE_FOR_KNOWN_DOMAIN) {
        initiatePiaFromPopup(request.data);
    }
    else if (request.messageType == MessageKeys.PIA_CREATE_AUTH_WINDOW) {
        isClientAuthorized(sender.tab.id, request.data.serverUrl, request.data.browserURL, ((initialSender, authorizationResult) => {
            if (!authorizationResult) {
                let serverDetails = request.data;
                if (serverDetails != null) {
                    createAuthWindow(initialSender, serverDetails.serverUrl);
                }
            }
            else {
                sendMessageToContentScript(initialSender, MessageKeys.PIA_CREATE_AUTH_WINDOW_CLOSED, {});
            }
        }));
    }
    else if (request.messageType === MessageKeys.PIA_REQUEST_TAB_SETTINGS) {
        ensurePiaTabInfo(request.data.ticketNumber, async function (settings) {
            await sendMessageToContentScript(sender.tab.id, MessageKeys.PIA_REQUEST_TAB_SETTINGS_REPLY, settings);
        });
    }
    else {
        if (request.messageType == MessageKeys.CHECK_TAB_INITIAL_STATUS || request.messageType == MessageKeys.PIA_START_REQUEST_FROM_POPUP) {
            let ticketNumber = null;
            if (request.data != undefined) {
                ticketNumber = request.data.ticketNumber;
            }
            if (ticketNumber) {
                ensurePiaTabInfo(ticketNumber, async (tabSetting) => {
                    await sendMessageToContentScript(sender.tab.id, MessageKeys.SEND_TAB_SETTINGS, tabSetting);
                });
            }
            else {
                if (request.messageType == MessageKeys.PIA_START_REQUEST_FROM_POPUP) {
                    displayInitialPiaWindow(sender.tab.id);
                }
            }
        }
        else if (request.messageType == MessageKeys.PIA_COOKIE_INFO_QUERY) {
            notifyLoginSessionInfo(sender.tab.id, request.data.serverAddress);

        }
        else if (request.messageType == MessageKeys.PIA_CLOSE_AUTH_WINDOW_REQUEST) {
            if (authWindowId != null) {
                closeTab(authWindowId, function () {});
            }
        }
    }

    return true;
});

const displayInitialPiaWindow = async (tabid) => {
    await ensurePiaTabInfo(-1, async function (currentSettings) {
        let cookieExists = false;
        if (currentSettings) {
            cookieExists = currentSettings.cookieExists;
        }
        await sendMessageToContentScript(tabid, "PIA_START_APP_INITIAL_INFO", { cookieExists: cookieExists });
    });
}
const initiatePiaFromPopup = async function (domainData) {
    await addCustomDomainSupport(domainData);
    await addDomain(domainData);
    await displayInitialPiaWindow(domainData.tabId);
}

chrome.windows.onRemoved.addListener(async function (windowid, removeInfo) {
    if (authWindowId != null && windowid == authWindowId) {
        authWindowId = null;
        await sendMessageToContentScript(authParentWindowTabId, MessageKeys.PIA_CREATE_AUTH_WINDOW_CLOSED, {});
    }
});

chrome.tabs.onUpdated.addListener(async function (tabid, changeInfo, tab) {
    if (changeInfo.status === 'complete') {
        // When user performs any action on the newly opened auth tab, it should not be close by the timer running in content script.
        // Therefore, whenver user action on this windows is detected, reset the timer.
        try {
            if (authWindowId != null && authWindowId == tab.windowId) {
                await sendMessageToContentScript(authParentWindowTabId, MessageKeys.PIA_AUTH_WINDOW_UPDATED_REPLY, {});
            }
            else {
                await sendMessageToContentScript(tabid, MessageKeys.PIA_ADDRESSBAR_VALUE_CHANGED, {});
            }
        }
        catch (error) {
            console.log(error);
        }
    }
});
