const PIA_IFRAME_CONTAINER_ID = "pia-iframe-container-id";
const PIA_SETTING_DIV_ID = "pia-settings-div-id";
const PIA_MODAL_DIV = "pia-iframe-modal-div";
const PIA_IFRAME_ID = "pia-iframe-id";
const PIA_HEADER_SPACER_ID = 'pia-frame-window-header-spacer'
const PIA_LOADING_DIV ='pia-loading-div'
const PIA_HEADER_SETTINGS = "&#9881;";
const PIA_HEADER_MINIMIZE = "&#9472;";
const PIA_HEADER_MAXIMIZE = "&#9634;";
const PIA_HEADER_CLOSE = "&#10761;";
const PIA_SIGN_IN_BUTTON = "pia-signin-button";

//Note:
// The same set of keys is available in common.js file. The keys are duplicated because of the limitation in chrome extension
// that does not allow content_script.js file to import individual modules. Any changes to the below key may require corresponding changes in
// common.js file.
const CHECK_CONTENT_SCRIPT_LOADED = "CHECK_CONTENT_SCRIPT_LOADED";
const PIA_START_REQUEST_FROM_POPUP = "PIA_START_REQUEST_FROM_POPUP";
const CHECK_TAB_INITIAL_STATUS = "CHECK_TAB_INITIAL_STATUS";
const PIA_START_APP_INITIAL_INFO = "PIA_START_APP_INITIAL_INFO";
const PIA_COOKIE_INFO_QUERY = "PIA_COOKIE_INFO_QUERY";
const PIA_COOKIE_INFO_QUERY_REPLY = "PIA_COOKIE_INFO_QUERY_REPLY";
const PIA_CREATE_AUTH_WINDOW = "PIA_CREATE_AUTH_WINDOW";
const PIA_CREATE_AUTH_WINDOW_CLOSED_REPLY = "PIA_CREATE_AUTH_WINDOW_CLOSED";
const PIA_CREATE_AUTH_WINDOW_CREATED_REPLY = "PIA_CREATE_AUTH_WINDOW_CREATED";
const PIA_CLOSE_AUTH_WINDOW_REQUEST = "PIA_CLOSE_AUTH_WINDOW_REQUEST";
const PIA_AUTH_WINDOW_UPDATED_REPLY = "PIA_AUTH_WINDOW_UPDATED_REPLY";
const PIA_ADDRESSBAR_VALUE_CHANGED = "PIA_ADDRESSBAR_VALUE_CHANGED";
const PIA_REQUEST_TAB_SETTINGS = "PIA_REQUEST_TAB_SETTINGS";
const PIA_REQUEST_TAB_SETTINGS_REPLY = "PIA_REQUEST_TAB_SETTINGS_REPLY";
const SEND_TAB_SETTINGS = "SEND_TAB_SETTINGS";

const PIA_EMBED_ACTIVITY_LOG_FULL_SCREEN = "PIA_ACTIVITY_LOG_FULL_SCREEN";
const PIA_EMBED_ACTIVITY_LOG_MINIMIZED_SCREEN = "PIA_ACTIVITY_LOG_MINIMIZED_SCREEN";

const HEADER_BUTTON_ACTION_TYPE_SETTINGS = "settings";
const HEADER_BUTTON_ACTION_TYPE_RESIZE = "minmax";
const HEADER_BUTTON_ACTION_TYPE_CLOSE = "close";
const SETTING_BUTTON_ACTION_TYPE_CLOSE = "close-setting";
const SETTING_BUTTON_ACTION_TYPE_UPDATE = "update-setting";
const customDomainKey = "PiaCustomDomains";

const AUTH_WINDOW_WAIT_DURATION = 60; // Seconds

let isPiaLoadingInProgress = false;

const PIA_API_EXTENSION_PATH = "/Extension/PiaEmbed/";
const PIA_API_PATH_REGEX = new RegExp("\\.pia\\.ai\\/?$")
const HALO_LOGIN_PATH = "/auth/Account/Login";
const AUTOTASK_LOGIN_PATH = "/Authentication.mvc";

// SETTING BOX TEXTS
const PIA_SETUP_SERVER_ADDRESS_TEXT = "Setup Server address";
const PIA_SETUP_START_PIA_TEXT = "Click to start using Pia";
const PIA_SETUP_SELECT_TICKET_TEXT = "Open a ticket";
const PIA_SETUP_EXISTING_DOMAIN_TEXT = "Ticketing Systen:";
const DEFAULT_TICKETING_SYSTEM_OPTION = 'HaloPSA';

// SUPPORTED TICKETING SYSTEMS
// Note: The values in this constant are duplicated between content_script.js and common.js 
//       due to a constraint that prevents content_script.js from being a module
//       and using the import/export functionality.
const SupportedTicketingSystem = {
    Autotask: 'Autotask',
    HaloPSA: 'HaloPSA',
    ServiceNow: 'ServiceNow'
}

const SupportedTicketingSystemOptions = [
    { label: 'Autotask', value: 'Autotask' },
    { label: 'HaloPSA', value: 'HaloPSA' },
    { label: 'ServiceNow', value: 'ServiceNow' }
];

// PIA EXTENSION MODES
const PIA_EXT_MODE_MINIMISED = "PIA_MODE_MINIMISED";
const PIA_EXT_MODE_SETTINGS = "PIA_MODE_SETTINGS";
const PIA_EXT_MODE_CHATBOT = "PIA_MODE_CHATBOT";

// STORAGE KEYS
const STORAGE_EXTENSION_POSITION = "PiaExtensionPosition";


// FOR UI
let HEADER_INDEX = null;
//
const appCookiesName = [".AspNetCore.cookiet1C1", ".AspNetCore.cookiet1C2", ".AspNetCore.cookiet1", ".AspNetCore.CookiesC1", ".AspNetCore.Cookies", ".AspNetCore.CookiesC2"];
let authWindow = null;
let currentTabSessings = {
    tabId: null,
    ticketNumber: null,
    autoLunch: true,
    cookieExists: false,
    serverUrl: "",
    ticketingSystem: ""
}
let initialStatusSent = false;
let piaLuncher = null;
let piaCookieObserver = null;
let loginStartedTime = null;
let sendInitialCheckRequestMessageKey = null; // this tracks what message to send after retrieving the tab settings

// global variable for tracking the extension mode
let LAST_KNOWN_EXTENSION_MODE = PIA_EXT_MODE_MINIMISED;

const getServerAddress = function () {
    if (currentTabSessings.serverUrl == undefined) {
        return null;
    }
    let serverUrl = currentTabSessings.serverUrl.trim();
    if (serverUrl.endsWith("/")) {
        serverUrl = serverUrl.slice(0, -1);
    }
    return serverUrl;
}

const createSettingView = async function () {
    const piaServerAddress = currentTabSessings.serverUrl != undefined ? currentTabSessings.serverUrl : "";
    let piaTicketNumber = await getTicketNumber();
    if (piaTicketNumber == undefined || isNaN(piaTicketNumber)) {
        piaTicketNumber = currentTabSessings.ticketNumber;
    }
    const parentDiv = $("#" + PIA_IFRAME_CONTAINER_ID);

    LAST_KNOWN_EXTENSION_MODE = PIA_EXT_MODE_SETTINGS;
    ExtensionPositionManager.updateExtensionPosition(PIA_EXT_MODE_SETTINGS);

    const settingDiv = $("<div id='" + PIA_SETTING_DIV_ID + "'></div>");
    const addressDiv = $("<div class='user-input'></div>");
    const serverAddressLabel = $("<label for='pia-settingview-serverurl'>Server Address:</label>");
    const serverAddressValue = $("<input type='text' id='pia-settingview-serverurl'></input>");
    const serverAddressValidationErrorDiv = $("<div id='pia-settingview-serverurl-validation-div'></div>");
    const validationError = $("<small id='pia-settingview-serverurl-error'></small>");
    serverAddressValue.val(piaServerAddress);
    addressDiv.append(serverAddressLabel);
    addressDiv.append(serverAddressValue);
    serverAddressValidationErrorDiv.append(validationError);
    const ticketDiv = $("<div class='user-input'></div>");
    const ticketLabel = $("<label for='pia-settingview-ticket'>Ticket Number:</label>");
    const ticketValue = $('<input/>').attr({ type: 'text', id: 'pia-settingview-ticket', name: 'ticketinput' });
    ticketValue.val(piaTicketNumber);

    const registeredSystemDropdownValue = await getTicketingSystemNameForDropdown();

    const registeredSystemDiv = $("<div class='user-input'></div>");
    const registeredSystemLabel = $("<label for='pia-settingview-ticket'>Ticketing System:</label>");
    const registeredSystemValue = $('<select/>').attr({ id: 'pia-settingview-system', name: 'registeredSystem' });

    SupportedTicketingSystemOptions.forEach(s => {
        const systemOptions = new Option(s.label, s.value, s.value == DEFAULT_TICKETING_SYSTEM_OPTION, s.value == registeredSystemDropdownValue);
        registeredSystemValue.append(systemOptions);
    });
    ticketValue.on('click', function (e) {
        e.target.focus();
    });
    serverAddressValue.on('click', function (e) {
        e.target.focus();
    });

    const autoLunchDiv = $("<div class='user-input'></div>");
    const autoLunch = $('<input/>').attr({ type: 'checkbox', id: 'pia-settingview-autolunch', name: 'autolunchPia' });
    const autoLunchtLabel = $("<label for='pia-settingview-autolunch'>Auto Launch Pia</label>");

    autoLunchDiv.append(autoLunch);
    autoLunchDiv.append(autoLunchtLabel);
    
    const settingActionButtonDiv = $("<div style='float: right;'></div>");
    const settingUpdate = $("<button class='pia-settings-action-buttons'>Update</button>");
    const settingClose = $("<button class='pia-settings-action-buttons'>Close</button>");
    autoLunch.prop('checked', currentTabSessings.autoLunch);
    settingActionButtonDiv.append(settingUpdate);
    settingActionButtonDiv.append(settingClose);

    settingUpdate.on('click', async function (e) {
        if (!/^https?:\/\//.test(serverAddressValue.val())) {
            document.getElementById('pia-settingview-serverurl-error').textContent = "Missing protocol http:// or https://";
        }
        else if (!PIA_API_PATH_REGEX.test(serverAddressValue.val())) {
            document.getElementById('pia-settingview-serverurl-error').textContent = "Server address must end in .pia.ai";
        }
        else {
            document.getElementById('pia-settingview-serverurl-error').textContent = "";
            $("#" + PIA_IFRAME_CONTAINER_ID).removeClass("pia-frame-setting");
            const reload = (currentTabSessings.ticketNumber != undefined && currentTabSessings.ticketNumber != ticketValue.val());
            currentTabSessings.serverUrl = serverAddressValue.val();
            currentTabSessings.ticketNumber = ticketValue.val();
            currentTabSessings.autoLunch = autoLunch.prop('checked');
            currentTabSessings.ticketingSystem = await getCurrentTicketingSystem();
            updateSettings();
            await updateSystemRegistered(registeredSystemValue.val());
        }
    });
    settingClose.on('click', function (e) {
        $("#" + PIA_IFRAME_CONTAINER_ID).removeClass("pia-frame-setting");
    });

    ticketDiv.append(ticketLabel);
    ticketDiv.append(ticketValue);
    registeredSystemDiv.append(registeredSystemLabel);
    registeredSystemDiv.append(registeredSystemValue);
    settingDiv.append(addressDiv);
    settingDiv.append(serverAddressValidationErrorDiv);
    settingDiv.append(ticketDiv);
    settingDiv.append(registeredSystemDiv);
    settingDiv.append(autoLunchDiv);
    settingDiv.append(settingActionButtonDiv);
    parentDiv.append(settingDiv);
}

const showCurrentSettings = function () {
    // Check if pia setting div is available
    const settingDiv = $("#" + PIA_SETTING_DIV_ID);
    if (settingDiv.length) {
        settingDiv.remove();
    }
    createSettingView();
}

const addPiaToScreen = function () {
    if (isPiaLoadingInProgress) {
        return;
    }
    isPiaLoadingInProgress = true;
    if (!verifyPiaServerAddress()) {
        return;
    }
    piaLuncher = setTimeout(function () {
        reorderHeaderWithIframe(true);
        const frameSource = getServerAddress() + PIA_API_EXTENSION_PATH + "?ticketId=" + currentTabSessings.ticketNumber;
        const piaChatWindowIndex = calculatePiaChatIndex();
        let iframe = $("#" + PIA_IFRAME_ID);
        if (iframe.length <= 0) {
            iframe = $("<iframe src='" + frameSource + "' id='" + PIA_IFRAME_ID + "' class='pia-iframe' style='width:100%;height:100%;z-index:" + (piaChatWindowIndex) + "';></iframe>")
            $("#" + PIA_IFRAME_CONTAINER_ID).append(iframe);
        }
        else {
            iframe.attr('src', frameSource);
        }
        $("#" + PIA_SIGN_IN_BUTTON).hide();
        const containerid = "#" + PIA_MODAL_DIV;
        const piaFrame = $(containerid);
        $("#" + PIA_IFRAME_CONTAINER_ID).removeClass("pia-frame-window-before");
        $("#" + PIA_IFRAME_CONTAINER_ID).addClass("pia-frame-window-after");
        piaFrame.toggle(100);
        isPiaLoadingInProgress = false;
    }, 1000);

}

const verifyPiaServerAddress = function () {
    const piaServerAddress = getServerAddress();
    if (piaServerAddress == undefined || piaServerAddress.trim() == "") {
        return false;
    }
    return true;
}

const checkIfPiaFrameExists = function () {
    let piaFrameExists = document.getElementById(PIA_IFRAME_CONTAINER_ID) !== null;
    let loadingDivExists = document.getElementById(PIA_LOADING_DIV) !== null;
    return piaFrameExists || loadingDivExists;
}
const minimizePia = function () {
    // Only allow when pia is loaded
    if ($("#" + PIA_IFRAME_ID).length == 0) {
        return;
    }

    LAST_KNOWN_EXTENSION_MODE = PIA_EXT_MODE_MINIMISED;
    ExtensionPositionManager.updateExtensionPosition(PIA_EXT_MODE_MINIMISED);
    updateStartPiaButton();

    $("#pia-frame-window-header").removeClass('pia-modal-header-afterload');
    $(".pia-frame-window-header-minimize").removeClass("pia-modal-header-minize-afterload");
    $(".pia-frame-window-header-settings").show();
    $(".pia-frame-window-header-close").show();
    $("#" + PIA_HEADER_SPACER_ID).show();
    const t = $("#" + PIA_IFRAME_ID);
    $("#" + PIA_IFRAME_ID).hide();
    $("#" + PIA_IFRAME_CONTAINER_ID).removeClass("pia-frame-window-after");
    $("#" + PIA_IFRAME_CONTAINER_ID).addClass("pia-frame-window-before");
    $("#pia-signin-button").show();
    $(".pia-frame-window-header-minimize").html(PIA_HEADER_MAXIMIZE);
}
const maximizePia = function () {
    LAST_KNOWN_EXTENSION_MODE = PIA_EXT_MODE_CHATBOT;
    ExtensionPositionManager.updateExtensionPosition(PIA_EXT_MODE_CHATBOT);
    $(".pia-frame-window-header-minimize").html(PIA_HEADER_MINIMIZE);
    $("#pia-frame-window-header").addClass('pia-modal-header-afterload');
    $("#pia-frame-window-header").css({ 'z-index': HEADER_INDEX + 1 });
    $(".pia-frame-window-header-minimize").addClass("pia-modal-header-minize-afterload");
    $("#" + PIA_HEADER_SPACER_ID).hide();
    $(".pia-frame-window-header-settings").hide();
    $(".pia-frame-window-header-close").hide();
    $("#" + PIA_IFRAME_CONTAINER_ID).addClass("pia-frame-window-after");
    $("#" + PIA_IFRAME_CONTAINER_ID).removeClass("pia-frame-window-before");
    $("#" + PIA_IFRAME_ID).show();
    $("#pia-signin-button").hide();
    $("#pia-settings-div-id").hide();
    $("#" + PIA_IFRAME_CONTAINER_ID).removeClass("pia-frame-setting");
}
const reorderHeaderWithIframe = function (frameLoaded) {
    if (frameLoaded) {
        maximizePia();
    }
}

const withinPSALoginWindow = function () {
    let currentWindow = window.location.href;
    return currentWindow.indexOf(AUTOTASK_LOGIN_PATH) !== -1 || currentWindow.indexOf(HALO_LOGIN_PATH) !== -1
}

const verifyPiaFrameAndLunch = async function (lunchPiaWindow, showPiaWindow) {
    if (!await getTicketNumber() && !showPiaWindow) {
        return;
    }
    if (checkIfPiaFrameExists()) {
        return;
    }

    if (withinPSALoginWindow()) {
        return;
    }

    const loadingDiv = document.createElement('div');
    loadingDiv.id = PIA_LOADING_DIV;
    document.body.appendChild(loadingDiv);

    // pia extension starts minimised
    LAST_KNOWN_EXTENSION_MODE = PIA_EXT_MODE_MINIMISED; 
    
    const zindex = getHighestZIndex();
    const parentDiv = document.createElement('div');
    if (HEADER_INDEX == null) {
        HEADER_INDEX = zindex + 1;
    }

    parentDiv.className = "pia-frame-window-before";
    parentDiv.id = PIA_IFRAME_CONTAINER_ID;
    parentDiv.style.zIndex = HEADER_INDEX;

    const modalDiv = document.createElement('div');
    modalDiv.className = "modal-card";
    modalDiv.style.width = "100%";

    const headerDiv = document.createElement('div');
    headerDiv.id = "pia-frame-window-header";

    // Add title and 
    const header = document.createElement('p');
    header.innerHTML = "<b>Pia Window</b>";
    header.className = 'pia-frame-window-header';

    const signInbutton = document.createElement('button');
    // If start pia was clicked from the extension but no ticket is found in the URL, disable the click event and
    // set the label to ticket not found
    signInbutton.id = PIA_SIGN_IN_BUTTON;
    signInbutton.className = "pia-start-using-button";
    updateStartPiaButton(signInbutton);
    headerDiv.appendChild(signInbutton);

    const settingsButton = document.createElement('button');
    settingsButton.innerHTML = PIA_HEADER_SETTINGS;
    settingsButton.className = 'pia-frame-window-header-action-buttons pia-frame-window-header-settings';
    settingsButton.onclick = function (e) {
        performHeaderButtonAction(HEADER_BUTTON_ACTION_TYPE_SETTINGS);
    };

    const minimizeButton = document.createElement('button');
    minimizeButton.className = 'pia-frame-window-header-action-buttons pia-frame-window-header-minimize';
    minimizeButton.innerHTML = PIA_HEADER_MINIMIZE;
    minimizeButton.onclick = function (e) {
        performHeaderButtonAction(HEADER_BUTTON_ACTION_TYPE_RESIZE);
    };
    const closeButton = document.createElement('button');
    closeButton.className = 'pia-frame-window-header-action-buttons pia-frame-window-header-close';
    closeButton.innerHTML = PIA_HEADER_CLOSE;
    closeButton.onclick = function (e) {
        performHeaderButtonAction(HEADER_BUTTON_ACTION_TYPE_CLOSE);
    };

    const spacer = document.createElement('div');
    spacer.id = PIA_HEADER_SPACER_ID;

    //headerDiv.appendChild(header);
    headerDiv.appendChild(spacer);
    headerDiv.appendChild(minimizeButton);
    headerDiv.appendChild(settingsButton);
    headerDiv.appendChild(closeButton);
    // Add body
    const bodyDiv = document.createElement('section');
    bodyDiv.style.height = "100px";
    bodyDiv.id = PIA_MODAL_DIV;

    modalDiv.appendChild(headerDiv);

    const coverLayer = document.createElement('div');
    coverLayer.id = "pia-frame-cover-layer";
    coverLayer.style.zIndex = getHighestZIndex() + 10;

    parentDiv.appendChild(coverLayer);
    //modalDiv.appendChild(bodyDiv);
    parentDiv.appendChild(modalDiv);

    // load save extension position from storage
    const savedPosition = await ExtensionPositionManager.loadExtensionPositionFromStorage();

    // update extension position global variable before calculations
    ExtensionPositionManager.setTrackedPosition(savedPosition);

    const nextPosition = ExtensionPositionManager.getNextExtensionPositionClippedToWindowBoundary(PIA_EXT_MODE_MINIMISED);

    // update extension position global variable
    ExtensionPositionManager.setTrackedPosition(nextPosition);

    // position the parent div
    parentDiv.style.left = ExtensionPositionManager.getTrackedPosition().left + "px";
    parentDiv.style.top = ExtensionPositionManager.getTrackedPosition().top + "px";

    // now add our extension div to the DOM
    document.body.appendChild(parentDiv);

    // Persist the calculated position to storage
    ExtensionPositionManager.saveExtensionPositionToStorage();

    // Register draggable feature
    const currentTicketingSystem = await getCurrentTicketingSystem();
    registerControlToBeDraggable(parentDiv, currentTicketingSystem);

    // Register activity log fullscreen event
    // Register window resize event
    if (window.addEventListener) {
        window.addEventListener("message", onActivityLogSizeChanged, false);
        window.addEventListener("resize", onWindowResize);
    }
    else if (window.attachEvent) {
        window.attachEvent("onmessage", onActivityLogSizeChanged, false);
        window.attachEvent("resize", onWindowResize);
    }


    // When auto lunching Pia, make sure the url has the ticket info (id)
    const ticketId = await getTicketNumber();
    if (ticketId != undefined && lunchPiaWindow != undefined && lunchPiaWindow == true) {
        startPiaProcess();
    }
}

const throttle = function (callback, delay) {
    let executeCallback = true;

    return function () {
        if (executeCallback) {
            // set the current context (this keyword) and pass through all arguments (arguments keyword) to the callback function
            // see: https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Function/apply
            callback.apply(this, arguments);

            // prevent the current function from being invoked until this flag is flipped back to true
            executeCallback = false;

            setTimeout(function () {
                // flip the flag back to true after waiting for "delay" milliseconds
                executeCallback = true;
            }, delay);
        }
    }
}

const onWindowResize = throttle(function () {
    ExtensionPositionManager.updateExtensionPosition(LAST_KNOWN_EXTENSION_MODE);
}, 100);

const startPiaProcess = function () {
    if (currentTabSessings.ticketNumber == null || currentTabSessings.ticketNumber.trim() == "") {
        return;
    }
    if (!verifyPiaServerAddress()) {
        return;
    }
    if (authWindow != null) {
        authWindow = null;
    }

    if (piaCookieObserver != null) {
        clearInterval(piaCookieObserver);
    }
    isPiaLoadingInProgress = true;
    createAuthWindow(getServerAddress());
}

const cleanUpLoginResources = function () {
    clearInterval(piaCookieObserver);
    loginStartedTime = null;
}

const performHeaderButtonAction = function (actionType) {
    if (actionType == HEADER_BUTTON_ACTION_TYPE_SETTINGS) {
        $("#" + PIA_IFRAME_ID).hide();
        showCurrentSettings();
        $("#" + PIA_IFRAME_CONTAINER_ID).addClass("pia-frame-setting");
    }
    else if (actionType == HEADER_BUTTON_ACTION_TYPE_RESIZE) {
        let containerid = "#" + PIA_MODAL_DIV;
        let piaFrame = $(containerid);

        const minimizeHeaderSelectedOption = $(".pia-frame-window-header-minimize");
        if (encodeURI(minimizeHeaderSelectedOption.html()) == '%E2%94%80') {
            minimizePia();

        }
        else {
            maximizePia();
        }
        piaFrame.toggle(100);
    }
    else if (HEADER_BUTTON_ACTION_TYPE_CLOSE) {
        let containerid = "#" + PIA_IFRAME_CONTAINER_ID;
        let piaFrame = $(containerid);
        piaFrame.remove();
        let loadingDiv = document.getElementById(PIA_LOADING_DIV);
        if (loadingDiv !== null) {
            loadingDiv.remove();
        }
    }
}

const getHighestZIndex = function () {
    return 2147483647;
}

const sendMessageToBackground = async function (messageType, additionalData) {
    await chrome.runtime.sendMessage({ messageType: messageType, data: additionalData }, function (response) {
    });
}

const getParameterByName = function (name, url = window.location.href) {
    name = name.replace(/[\[\]]/g, '\\$&');
    const regex = new RegExp('[?&]' + name + '(=([^&#]*)|&|#|$)'),
        results = regex.exec(url);
    if (!results) return null;
    if (!results[2]) return '';
    return decodeURIComponent(results[2].replace(/\+/g, ' '));
}

async function sendInitialCheckRequest(messageKey) {
    sendInitialCheckRequestMessageKey = messageKey;
    let ticketNumber = await getTicketNumber();
    await sendMessageToBackground(PIA_REQUEST_TAB_SETTINGS, { ticketNumber: ticketNumber });
}

const checkIfLoginSessionExists = async function () {
    await sendMessageToBackground(PIA_COOKIE_INFO_QUERY, { serverAddress: currentTabSessings.serverUrl });
}
const updateSettings = function () {
    const setting = {
        ticketNumber: currentTabSessings.ticketNumber,
        autoLunch: currentTabSessings.autoLunch,
        serverUrl: currentTabSessings.serverUrl
    };
    chrome.storage.sync.set({ PiaTabInfo: setting }, async function () {
        if (currentTabSessings.autoLunch == true && currentTabSessings.ticketNumber && currentTabSessings.ticketNumber != "" && currentTabSessings.serverUrl && currentTabSessings.serverUrl != "" && currentTabSessings.cookieExists && currentTabSessings.ticketingSystem != "") {
            await checkIfLoginSessionExists();
        }
        else {
            updateStartPiaButton();
        }
    });
}

const createAuthWindow = async function (serverUrl, authPath) {
    await sendMessageToBackground(PIA_CREATE_AUTH_WINDOW, { serverUrl: serverUrl, browserURL: window.location.href });
}

/**
 * Returns the current ticketing system when we are on known addresses
 * Otherwise will attempt to return the ticketing system based on configured custom domain details for the current domain
 * Otherwise will return an empty string
 * @returns {Promise<string>}
 */
const getCurrentTicketingSystem = async function () {
    let serverURL = window.location.href;
    let currentTicketingSystem = "";
    let collectedFromExistingDetails = false;

    if (serverURL.match('halopsa.com')) {
        currentTicketingSystem = SupportedTicketingSystem.HaloPSA;
    }
    else if (serverURL.match('service-now.com')) {
        let incidentPos = serverURL.indexOf("/record/incident/");

        if (incidentPos !== -1) {
            currentTicketingSystem = SupportedTicketingSystem.ServiceNow;
        }
    }
    else if (serverURL.indexOf("TicketDetail.mvc") !== -1 && serverURL.indexOf("Authentication.mvc") === -1) {
        currentTicketingSystem = SupportedTicketingSystem.Autotask;
    }
    else {
        let existingCustomDomainInfo = await tryFetchExistingDomainDetail(serverURL);

        if (existingCustomDomainInfo != null) {
            currentTicketingSystem = existingCustomDomainInfo.system;
            collectedFromExistingDetails = true;
        }
    }

    if (currentTicketingSystem == SupportedTicketingSystem.Autotask && collectedFromExistingDetails) {
        return // There is no need for us to collect the autotask system via the cache
    }

    return currentTicketingSystem;
}

const getTicketNumberBasedOnTicketingSystem = function (url, ticketingSystem) {
    let ticketNumber = null;

    if (ticketingSystem === SupportedTicketingSystem.HaloPSA) {
        //e.g. https://piaai.halopsa.com/Tickets?area=1&mainview=myviews&viewid=1&selid=25&sellevel=1&selparentid=0&id=2626
        let haloPSATicketUrlRegex = new RegExp(getDomainFromUrl(url) + '/Ticket[s]?.*[?&]id=(\\d+)', 'i');
        let ticketMatch = url.match(haloPSATicketUrlRegex);

        if (ticketMatch && ticketMatch.length > 1) {
            ticketNumber = ticketMatch[1];
        }
    }
    else if (ticketingSystem === SupportedTicketingSystem.Autotask) {
        ticketNumber = getParameterByName("ticketId") || getParameterByName("TicketID");
    }
    else if (ticketingSystem === SupportedTicketingSystem.ServiceNow) {
        let incidentPos = url.indexOf("/record/incident/");

        if (incidentPos !== -1) {
            ticketNumber = url.substring(incidentPos + "/record/incident/".length, incidentPos + "/record/incident/".length + 32)
        }
    }

    return ticketNumber;
}

/**
 * @returns {Promise<string | null>}
 */
const getTicketNumber = async function () {
    let ticketNumber = null;
    let serverURL = window.location.href;
    let currentTicketingSystem = await getCurrentTicketingSystem();

    if (currentTicketingSystem !== "") {
        ticketNumber = getTicketNumberBasedOnTicketingSystem(serverURL, currentTicketingSystem);
    }

    return ticketNumber;
}

const updateStartPiaButton = function (signInbutton) {
    signInbutton = clearEventOfSignInButton(signInbutton);
    if (!currentTabSessings.serverUrl || currentTabSessings.serverUrl == "") {
        signInbutton.innerHTML = PIA_SETUP_SERVER_ADDRESS_TEXT;
        signInbutton.addEventListener('click', pickATicketHandler);
        return;
    }
    if (currentTabSessings.ticketNumber != null && currentTabSessings.ticketNumber != "" && currentTabSessings.ticketingSystem != null && currentTabSessings.ticketingSystem != "") {
        signInbutton.innerHTML = PIA_SETUP_START_PIA_TEXT;
        signInbutton.addEventListener('click', startPiaProcessHandler);
    }
    else {
        signInbutton.innerHTML = PIA_SETUP_SELECT_TICKET_TEXT;
        signInbutton.addEventListener('click', pickATicketHandler);
    }
}

// Events
const startPiaProcessHandler = function () {
    if (isPiaLoadingInProgress == false)
        startPiaProcess();
}

const pickATicketHandler = function () {
    performHeaderButtonAction(HEADER_BUTTON_ACTION_TYPE_SETTINGS);
}

const clearEventOfSignInButton = function (signInbutton) {
    if (!signInbutton) {
        signInbutton = document.getElementById(PIA_SIGN_IN_BUTTON);
    }
    signInbutton.removeEventListener('click', startPiaProcessHandler);
    signInbutton.removeEventListener('click', pickATicketHandler);
    return signInbutton;
}

// Make control draggable
const registerControlToBeDraggable = function (elmnt, ticketingSystem) {
    elmnt.onmousedown = dragMouseDown;
    elmnt.onmouseup = dragMouseUp;

    // the parent containment strategy prevents the extension from being dragged outside
    // the boundary of the element it's appended to.
    let containment = "parent";

    // AutoTask has broken <HTML> and <BODY> sizes.
    // The document containment strategy prevents the extension from being dragged outside
    // the visible window boundary.
    if (ticketingSystem === SupportedTicketingSystem.Autotask) {
        containment = "document";
    }

    // documentation: https://api.jqueryui.com/draggable/
    $(elmnt).draggable({ containment: containment });

    function dragMouseDown(e) {
        if (e.target.id == "pia-frame-window-header") {
            document.getElementById("pia-frame-cover-layer").style.display = "block";
        }
    }

    function dragMouseUp() {
        document.getElementById("pia-frame-cover-layer").style.display = "none";
        // stop moving when mouse button is released:
        document.onmouseup = null;
        document.onmousemove = null;

        // update position tracking variable
        const position = ExtensionPositionManager.getExtensionPosition();
        ExtensionPositionManager.setTrackedPosition(position);

        // persist current extension position in storage
        ExtensionPositionManager.saveExtensionPositionToStorage();
    }
}

const calculatePiaChatIndex = function () {
    let piaChatWindowIndex = getHighestZIndex() - 1;
    const headerDiv = $("#" + PIA_IFRAME_CONTAINER_ID);
    if (headerDiv.length) {
        const zindex = headerDiv.css("z-index");
        if (isNaN(zindex) == false) {
            piaChatWindowIndex = (zindex - 1);
        }
    }
    return piaChatWindowIndex;
}

const getDomainFromUrl = (url) => new URL(url).origin;

const getCurrentDomain = (currentTabUrl) => getDomainFromUrl(currentTabUrl) + '/*';

const tryFetchExistingDomainDetail = async function (url) {
    //POPUP_HTML
    const data = await chrome.storage.sync.get(customDomainKey);
    if (data === undefined || data.PiaCustomDomains === undefined) {
        return null;
    }
    const existingItem = data.PiaCustomDomains.filter(domain => (domain != null && domain.serverAddress == getCurrentDomain(url)));
    if (existingItem != null && existingItem.length > 0) {
        return existingItem[0];
    }
    return null;
}

const onActivityLogSizeChanged = function (event) {
    //Only listen event from the server
    if (event.origin != getServerAddress()) {
        return;
    }
    if (event.data == PIA_EMBED_ACTIVITY_LOG_FULL_SCREEN) {
        $('#pia-frame-window-header').addClass('on-activity-log-fullscreen');
    }
    else if (event.data == PIA_EMBED_ACTIVITY_LOG_MINIMIZED_SCREEN) {
        $('#pia-frame-window-header').removeClass('on-activity-log-fullscreen')
    }
}

/**
 * @returns {Promise<string>}
 */
const getTicketingSystemNameForDropdown = async () => {
    let serviceUrl = window.location.href;

    const systemRegex = {
        AUTOTASK_DOMAIN: "\\bhttps:\/\/.+.autotask.net\\b.+",
        HALOPSA_DOMAIN: "\\bhttps:\/\/.+.halopsa.com\\b.+",
        SERVICENOW_DOMAIN: "\\bhttps:\/\/.+.service-now.com\\b.+"
    }

    if (serviceUrl.match(systemRegex.AUTOTASK_DOMAIN)) {
        return SupportedTicketingSystem.Autotask;
    }
    if (serviceUrl.match(systemRegex.HALOPSA_DOMAIN)) {
        return SupportedTicketingSystem.HaloPSA;
    }
    if (serviceUrl.match(systemRegex.SERVICENOW_DOMAIN)) {
        return SupportedTicketingSystem.ServiceNow;
    }

    let customDomain = await getCurrentDomainFromStore(serviceUrl);

    if (customDomain != null) {
        return customDomain.system;
    }

    // default to HaloPSA if we haven't found a match
    return SupportedTicketingSystem.HaloPSA;
}


/**
 * @param {string} url
 * @returns {Promise<T|null>}
 */
const getCurrentDomainFromStore = async (url) => {
    let customDomainUrl = getCurrentDomain(url);
    const data = await chrome.storage.sync.get(customDomainKey);

    if (typeof (data) === "undefined" || typeof (data.PiaCustomDomains) === "undefined") {
        return null;
    }

    const existingItem = data.PiaCustomDomains.filter(domain => domain != null && domain.serverAddress == customDomainUrl);

    if (existingItem != null && existingItem.length > 0) {
        return existingItem[0];
    }

    return null;
};

const updateExistingDomainDetail = async function (newData) {
    const data = await chrome.storage.sync.get(customDomainKey);

    if (data === undefined || data.PiaCustomDomains === undefined) {
        return null;
    }

    const updatedList = data.PiaCustomDomains.map(domain => {
        if ((domain != null && domain.serverAddress == newData.serverAddress)) {
            domain.system = newData.system;
        }
        return domain;
    });

    return updatedList;
}

/**
 * @param {string} newSystemType
 * @returns {Promise<void>}
 */
const updateSystemRegistered = async function (newSystemType) {
    let currentUrl = getCurrentDomain(window.location.href);
    let domainData = await tryFetchExistingDomainDetail(currentUrl);

    if (domainData == null) {
        return;
    }

    domainData.system = newSystemType;

    chrome.storage.sync.get(customDomainKey).then(async (data) => {
        if (data != null && data.PiaCustomDomains != null) {
            let currentDomains = data.PiaCustomDomains;
            const existingDomainInfo = await tryFetchExistingDomainDetail(currentUrl);

            if (existingDomainInfo == null) {
                currentDomains.push(domainData);
            }
            else {
                //update ticketing system type
                const updatedDomains = await updateExistingDomainDetail(domainData);

                if (updatedDomains != null) {
                    currentDomains = updatedDomains;
                }
            }

            await chrome.storage.sync.set({ PiaCustomDomains: currentDomains });
        }
    });
}

// Chrome APi Start
// Any conditions that call the async method, need to return true. This will indicate that the calling section that the current operation is
// async and prevents the error Uncaught (in Promise) Error. So, the implementation does not await any call and immediately returns true at the end.
// https://developer.chrome.com/docs/extensions/develop/concepts/messaging
chrome.runtime.onMessage.addListener(async (message, sender, sendResponse) => {
    sendResponse({ response: "true" });
    const backgroundMessage = message.message;
    if (backgroundMessage.messageType == CHECK_CONTENT_SCRIPT_LOADED) {
        sendResponse(true);
    }
    const ticketNumber = await getTicketNumber();


    if (backgroundMessage.messageType === PIA_REQUEST_TAB_SETTINGS_REPLY) {
        // part of the loading process - retrieved tab settings

        // extract settings from message
        currentTabSessings = backgroundMessage.data;

        // continue the loading process
        await sendMessageToBackground(sendInitialCheckRequestMessageKey, { ticketNumber: ticketNumber });
    }
    else if (backgroundMessage.messageType == PIA_ADDRESSBAR_VALUE_CHANGED) {
        if (ticketNumber !== null && ticketNumber !== "") {
            if (currentTabSessings.ticketNumber == null || currentTabSessings.ticketNumber !== ticketNumber) {
                performHeaderButtonAction(HEADER_BUTTON_ACTION_TYPE_CLOSE);
            }
            sendMessageToBackground(CHECK_TAB_INITIAL_STATUS, { ticketNumber: ticketNumber });
        }
        else {
            performHeaderButtonAction(HEADER_BUTTON_ACTION_TYPE_CLOSE);
            currentTabSessings.ticketNumber = "";
        }
    }
    else if (backgroundMessage.messageType == PIA_START_REQUEST_FROM_POPUP) {
        sendInitialCheckRequest(PIA_START_REQUEST_FROM_POPUP);
    }
    else if (backgroundMessage.messageType == PIA_START_APP_INITIAL_INFO) {
        // This section is invoked when the PIA plugin is clicked.
        // Determine whether or not to show the PIA initial screen.
        //Now check if the current url has ticket id associated with it
        if (ticketNumber != null && !isNaN(ticketNumber)) {
            currentTabSessings.ticketNumber = ticketNumber;
        }
        const recentlyAddedItem = document.getElementById(PIA_IFRAME_CONTAINER_ID);
        if (recentlyAddedItem == undefined) {
            verifyPiaFrameAndLunch(false, true);
        }
        else {
            console.log("Pia already exists");
        }
    }
    else if (backgroundMessage.messageType == PIA_COOKIE_INFO_QUERY_REPLY) {
        currentTabSessings.cookieExists = backgroundMessage.data.cookieExists;
        isPiaLoadingInProgress = false;
        if (currentTabSessings.cookieExists) {
            cleanUpLoginResources();
            addPiaToScreen();
        }
    }
    else if (backgroundMessage.messageType == SEND_TAB_SETTINGS) {
        currentTabSessings = backgroundMessage.data;
        verifyPiaFrameAndLunch(currentTabSessings.autoLunch);
    }
    else if (backgroundMessage.messageType == PIA_CREATE_AUTH_WINDOW_CLOSED_REPLY) {
        checkIfLoginSessionExists();
    }
    else if (backgroundMessage.messageType == PIA_AUTH_WINDOW_UPDATED_REPLY) {
        // When user performs any action on the newly opened auth tab, allow additional time to complete the authentication.
        loginStartedTime = new Date();
    }
    else if (backgroundMessage.messageType == PIA_CREATE_AUTH_WINDOW_CREATED_REPLY) {
        //start the time here
        authWindow = backgroundMessage.data.windowId;
        cleanUpLoginResources();
        loginStartedTime = new Date();
        piaCookieObserver = setInterval(async function () {
            // If pia was clicked but no attempt to login was made, close the window after a min and reset settings
            const diff = (new Date() - loginStartedTime) / 1000;
            const minutes = Math.abs(Math.round(diff));
            if (minutes >= AUTH_WINDOW_WAIT_DURATION) {
                await sendMessageToBackground(PIA_CLOSE_AUTH_WINDOW_REQUEST);
                cleanUpLoginResources();
            }
            await checkIfLoginSessionExists();
        }, 1000);
    }
    return true;
});
// Chrome APi End

/**
 * Class that manages the position of the extension
 */
class ExtensionPositionManager {
    /**
     * Used to track the position of the extension
     * @type {{top: number, left: number}}
     */
    static #trackedPosition = {
        left: 0,
        top: 0,
    };
    
    constructor() {
        // defaults go here
    }

    /**
     * @param {{ left: number, top: number }} position
     */
    static setTrackedPosition(position) {
        this.#trackedPosition.left = position.left;
        this.#trackedPosition.top = position.top;
    }

    /**
     * Returns the tracked position of the extension
     * @returns {{top: number, left: number}}
     */
    static getTrackedPosition() {
        return this.#trackedPosition;
    }

    /**
     * @param {"PIA_MODE_MINIMISED" | "PIA_MODE_SETTINGS" | "PIA_MODE_CHATBOT"} mode
     * @returns {{top: number, left: number}}
     */
    static getNextExtensionPositionClippedToWindowBoundary(mode) {
        const extensionDimensions = this.getExtensionDimensions(mode);
        const windowDimensions = this.getWindowDimensions();

        const maxLeft = Math.max(0, windowDimensions.width - extensionDimensions.width);
        const maxTop = Math.max(0, windowDimensions.height - extensionDimensions.height);

        // calculate the updated extension position
        return {
            left: Math.min(this.#trackedPosition.left, maxLeft),
            top: Math.min(this.#trackedPosition.top, maxTop),
        };
    }

    /**
     * @param {"PIA_MODE_MINIMISED" | "PIA_MODE_SETTINGS" | "PIA_MODE_CHATBOT"} mode
     */
    static updateExtensionPosition(mode) {
        const nextPosition = this.getNextExtensionPositionClippedToWindowBoundary(mode);
        this.setTrackedPosition(nextPosition);

        // position the extension element
        const container = document.getElementById(PIA_IFRAME_CONTAINER_ID);
        container.style.left = this.#trackedPosition.left + "px";
        container.style.top = this.#trackedPosition.top + "px";

        // persist position to storage
        ExtensionPositionManager.saveExtensionPositionToStorage();
    }

    /**
     * @returns {{width: number, height: number}}
     */
    static getWindowDimensions() {
        return {
            width: window.innerWidth,
            height: window.innerHeight,
        };
    }

    /**
     * @returns {{top: number, left: number}}
     */
    static getInitialExtensionPosition() {
        const windowDimensions = this.getWindowDimensions();

        return {
            left: Math.round(windowDimensions.width * 0.7), // 70% across
            top: Math.round(windowDimensions.height * 0.7), // 70% down
        };
    }

    /**
     * @returns {{top: number, left: number}}
     */
    static getExtensionPosition() {
        const container = document.getElementById(PIA_IFRAME_CONTAINER_ID);
        const style = getComputedStyle(container);
        const left = parseInt(style.getPropertyValue("left").replace("px", ""));
        const top = parseInt(style.getPropertyValue("top").replace("px", ""));

        return {
            left: left,
            top: top
        };
    }

    /**
     * @param {"PIA_MODE_MINIMISED" | "PIA_MODE_SETTINGS" | "PIA_MODE_CHATBOT"} mode
     * @returns {{width: number, height: number}}
     */
    static getExtensionDimensions(mode) {
        // TODO: later, retrieve these values from CSS variables
        // e.g. getComputedStyle(document.body).getPropertyValue('--pia-container-height')

        if (mode === PIA_EXT_MODE_MINIMISED) {
            return { width: 300, height: 30 };
        }
        else if (mode === PIA_EXT_MODE_SETTINGS) {
            return { width: 300, height: 232 };
        }
        else if (mode === PIA_EXT_MODE_CHATBOT) {
            return { width: 734, height: 600 };
        }
    }

    /**
     * @returns {Promise<void>}
     */
    static async saveExtensionPositionToStorage() {
        await chrome.storage.local.set({ [STORAGE_EXTENSION_POSITION]: this.#trackedPosition });
    }

    /**
     * @param {(position: { left: number, top: number }) => unknown} callback
     */
    static async loadExtensionPositionFromStorage() {
        const result = await chrome.storage.local.get([STORAGE_EXTENSION_POSITION]);

        let position = this.getInitialExtensionPosition();

        // verify storage item
        const isValid = typeof (result) === "object"
            && typeof (result[STORAGE_EXTENSION_POSITION]) === "object"
            && result[STORAGE_EXTENSION_POSITION].left >= 0
            && result[STORAGE_EXTENSION_POSITION].top >= 0;

        if (isValid) {
            position.left = result[STORAGE_EXTENSION_POSITION].left;
            position.top = result[STORAGE_EXTENSION_POSITION].top;
        }

        return position;
    }
}
