// Note: The values in this constant are duplicated between content_script.js and common.js 
//       due to a constraint that prevents content_script.js from being a module
//       and using the import/export functionality.
export const SupportedTicketingSystem = {
    Autotask: 'Autotask',
    HaloPSA: 'HaloPSA',
    ServiceNow: 'ServiceNow'
}

//Note: 
// The same set of keys is available in content_script.js file. The keys are duplicated because of the limitation in chrome extension 
// that does not allow content_script.js file to import individual moduled. Any changes to the below key may require corresponding changes in 
// content_script.js file.
export const MessageKeys = {
    PIA_ADD_CUSTOM_TICKETING_SYSTEM: "PIA_ADD_CUSTOM_TICKETING_SYSTEM",
    PIA_INITIATE_FOR_KNOWN_DOMAIN: "PIA_INITIATE_FOR_KNOWN_DOMAIN",
    PIA_COOKIE_INFO_QUERY: "PIA_COOKIE_INFO_QUERY",
    PIA_COOKIE_INFO_QUERY_REPLY: "PIA_COOKIE_INFO_QUERY_REPLY",
    PIA_CREATE_AUTH_WINDOW: "PIA_CREATE_AUTH_WINDOW",
    PIA_CREATE_AUTH_WINDOW_CREATED: "PIA_CREATE_AUTH_WINDOW_CREATED",
    PIA_CREATE_AUTH_WINDOW_CLOSED: "PIA_CREATE_AUTH_WINDOW_CLOSED",
    PIA_CLOSE_AUTH_WINDOW_REQUEST: "PIA_CLOSE_AUTH_WINDOW_REQUEST",
    PIA_AUTH_WINDOW_UPDATED_REPLY: "PIA_AUTH_WINDOW_UPDATED_REPLY",
    PIA_ADDRESSBAR_VALUE_CHANGED: "PIA_ADDRESSBAR_VALUE_CHANGED",
    CHECK_TAB_INITIAL_STATUS: "CHECK_TAB_INITIAL_STATUS",
    PIA_START_REQUEST_FROM_POPUP: "PIA_START_REQUEST_FROM_POPUP",
    SEND_TAB_SETTINGS: "SEND_TAB_SETTINGS",
    CHECK_CONTENT_SCRIPT_LOADED: "CHECK_CONTENT_SCRIPT_LOADED",
    PIA_REQUEST_TAB_SETTINGS: "PIA_REQUEST_TAB_SETTINGS",
    PIA_REQUEST_TAB_SETTINGS_REPLY: "PIA_REQUEST_TAB_SETTINGS_REPLY"
}

export const SystemRegex = {
    AUTOTASK_DOMAIN : "\\bhttps:\/\/.+.autotask.net\\b.+",
    HALOPSA_DOMAIN : "\\bhttps:\/\/.+.halopsa.com\\b.+",
    SERVICENOW_DOMAIN : "\\bhttps:\/\/.+.service-now.com\\b.+"
}

export const POPUP_HTML = "AddCustomDomain.html";
export const customDomainKey = "PiaCustomDomains";
export const SCRIPT_UIQUE_ID = "PIA_CUSTOMSCRIPT_UNIQUE_ID";
export const appCookiesName = [".AspNetCore.cookiet1C1", ".AspNetCore.cookiet1C2", ".AspNetCore.cookiet1", ".AspNetCore.CookiesC1", ".AspNetCore.Cookies", ".AspNetCore.CookiesC2"];

/**
 * @param {string} serviceUrl 
 * @returns {{ticketingSystem: string, isCustomDomain: boolean}}
 */
export const getTicketingSystemInfoFromUrl = (serviceUrl) => {
    let result = {
        ticketingSystem: 'UNKNOWN',
        isCustomDomain: false
    }
    if (serviceUrl.match(SystemRegex.AUTOTASK_DOMAIN)) {
        result.ticketingSystem = SupportedTicketingSystem.Autotask;
        return result;
    }
    if (serviceUrl.match(SystemRegex.HALOPSA_DOMAIN)) {
        result.ticketingSystem = SupportedTicketingSystem.HaloPSA;
        return result;
    }
    if (serviceUrl.match(SystemRegex.SERVICENOW_DOMAIN)) {
        result.ticketingSystem = SupportedTicketingSystem.ServiceNow;
        return result;
    }
    result.isCustomDomain = true;
    return result;
}

/**
 * @param {string} url
 * @returns {null|string}
 */
export const getTicketingSystemType = function (url) {
    let currentServerUrl = location.hostname;
    if (!(url == null || url.trim() === "")) {
        currentServerUrl = url;
    }
    if (verifyAutotaskDomain(currentServerUrl)) {
        return SupportedTicketingSystem.Autotask;
    } 
    else if (verifyHaloPSADomain(currentServerUrl)) {
        return SupportedTicketingSystem.HaloPSA;
    } 
    else if (verifyServiceNowDomain(currentServerUrl)) {
        return SupportedTicketingSystem.ServiceNow;
    } 
    else {
        return null;
    }
}

/**
 * @param {string} currentTabUrl
 * @returns {string}
 */
export const getCurrentDomain = (currentTabUrl) => new URL(currentTabUrl).origin + '/*';

/**
 * @param {string} url
 * @returns {Promise<T|boolean>}
 */
export const isCurrentDomainAlreadyAdded = async (url) => {
    let customDomainUrl = getCurrentDomain(url);
    const data = await chrome.storage.sync.get(customDomainKey);
    if (typeof(data) === "undefined" || typeof(data.PiaCustomDomains) === "undefined") {
        return false;
    }
    let existingItem = data.PiaCustomDomains.filter(domain => domain != null && domain.serverAddress == customDomainUrl);
    if (existingItem != null && existingItem.length > 0) {
        return existingItem[0];
    }
    return false;
};

export const buildDomainObject = (serverAddress, system, isCustomDomain, tabId) => {
    // console.log(serverAddress, system, isCustomDomain, tabId)
    return {
        serverAddress: serverAddress,
        system: system,
        isCustomDomain: isCustomDomain,
        tabId: tabId
    }
}

export const getCurrentActiveTab = async () => {
    let query = { active: true, currentWindow: true };
    const [tab] = await chrome.tabs.query(query);
    // console.log(tab);
    return tab;
}

/**
 * @param {string} taburl
 * @returns {boolean}
 */
export const verifyAutotaskDomain = function (taburl) {
    let isAutotaskDomain = taburl.match(SystemRegex.AUTOTASK_DOMAIN);
    return !!(isAutotaskDomain && isAutotaskDomain.length > 0);
}

/**
 * @param {string} taburl
 * @returns {boolean}
 */
export const verifyHaloPSADomain = function (taburl) {
    let isHaloPSADomain = taburl.match(SystemRegex.HALOPSA_DOMAIN);
    return !!(isHaloPSADomain && isHaloPSADomain.length > 0);
}

/**
 * @param {string} taburl
 * @returns {boolean}
 */
export const verifyServiceNowDomain = function (taburl) {
    let isServiceNowDomain = taburl.match(SystemRegex.SERVICENOW_DOMAIN);
    return !!(isServiceNowDomain && isServiceNowDomain.length > 0);
}