import { getCurrentDomain, customDomainKey, SCRIPT_UIQUE_ID } from './common.js'

const permissionList = ["cookies", "storage", "activeTab", "scripting", "tabs"];
const cssLists = ["css/chrome-extension-v2.css"];
const scriptLists = ["js/lib/jquery-3.6.4.min.js", "js/lib/jquery-ui-1.13.2.min.js", "js/content_script.js"];

const updateExistingDomainDetail = async function (newData) {
    const data = await chrome.storage.sync.get(customDomainKey);
    
    if (typeof(data) === "undefined" || typeof(data.PiaCustomDomains) === "undefined") {
        return null;
    }
    
    const updatedList = data.PiaCustomDomains.map(domain => {
        if ((domain != null && domain.serverAddress == newData.serverAddress)) {
            domain.system = newData.system;
        }
        return domain;
    });
    
    return updatedList;
}

const tryFetchExistingDomainDetail = async function (url) {
    const data = await chrome.storage.sync.get(customDomainKey);

    if (typeof(data) === "undefined" || typeof(data.PiaCustomDomains) === "undefined") {
        return null;
    }

    const existingItem = data.PiaCustomDomains.filter(domain => (domain != null && domain.serverAddress == getCurrentDomain(url)));

    if (existingItem != null && existingItem.length > 0) {
        return existingItem[0];
    }

    return null;
}

const addDomain = (domaindata) => {
    return new Promise((resolve, reject) => {
        if (domaindata.isCustomDomain == false) {
            resolve();
        }

        let customDomainUrl = getCurrentDomain(domaindata.serverAddress);

        chrome.storage.sync.get(customDomainKey).then(async (data) => {
            if (data === undefined || data.PiaCustomDomains === undefined) {
                chrome.storage.sync.set({ PiaCustomDomains: [domaindata] });
                resolve();
            }
            else {
                let currentDomains = data.PiaCustomDomains;
                const existingDomainInfo = await tryFetchExistingDomainDetail(customDomainUrl);

                if (existingDomainInfo == null) {
                    currentDomains.push(domaindata);
                }
                else {
                    //update ticketing system type
                    const updatedDomains = await updateExistingDomainDetail(domaindata);

                    if (updatedDomains != null) {
                        currentDomains = updatedDomains;
                    }
                }

                await chrome.storage.sync.set({ PiaCustomDomains: currentDomains });
                resolve();
            }
        });
    });
};

const addContextMenu = async () => {
    chrome.contextMenus.create({
        id: "addPiaContext",
        title: "Add Pia Support to this site",
        type: 'normal',
        contexts: ['selection']
    });
};

const RequestChromePermission = (domainData) => {
    return chrome.permissions.request({
        permissions: permissionList,
        origins: [domainData.serverAddress]
    });
}

const InjectStyles = (domainData) => {
    return chrome.scripting.insertCSS({
        target: { tabId: domainData.tabId },
        files: cssLists
    })
}

const ExecuteChromeScripts = (domainData) => {
    return chrome.scripting.executeScript({
        target: { tabId: domainData.tabId },
        files: scriptLists
    });
}

const addCustomDomainSupport = (domainData) => {
    return new Promise(async (resolve, reject) => {

        if (domainData.isCustomDomain == false) {
            return resolve();
        }

        RequestChromePermission(domainData).then((result) => {
            if (result == true) {
                InjectStyles(domainData).then(async () => {
                    let registrationResult = await registerContentScript(domainData.serverAddress, [domainData.serverAddress]);
                    if (registrationResult == true) {
                        const executionResult = await ExecuteChromeScripts(domainData);
                        resolve(executionResult);
                    }
                    else {
                        resolve();
                    }
                });
            }
        }).catch((requestError) => {
            console.error(requestError);
            const error = requestError;
        });

    });
};

const registerContentScript = async (scriptId, matches) => {
    try {
        const existingRegisterationRecord = await chrome.scripting.getRegisteredContentScripts({
            ids: [scriptId]
        });
        if (existingRegisterationRecord.length > 0) {
            return existingRegisterationRecord[0];
        }
        await chrome.scripting.registerContentScripts([
            {
                id: scriptId,
                matches: matches,
                js: scriptLists,
                css: cssLists
            }
        ]);
        return true;

    }
    catch (error) {
        return false;
    }
}

export { addDomain, addContextMenu, addCustomDomainSupport, getCurrentDomain }