import {
    isCurrentDomainAlreadyAdded,
    MessageKeys,
    SystemRegex,
    getTicketingSystemInfoFromUrl,
    getCurrentDomain,
    buildDomainObject,
    getCurrentActiveTab
} from './common.js'

/**
 * @type {HTMLSelectElement}
 */
const systemTypeElement = document.getElementById("ticketing_system_type");

/**
 * @type {HTMLButtonElement}
 */
const addToPiaElement = document.getElementById("add_pia_support");

const initiatePiaForOperation = async function (existingCallback, newCallback) {
    let tab = await getCurrentActiveTab();
    let taburl = tab.url;
    let tabid = tab.id;
    let ticketingSystemInfo = getTicketingSystemInfoFromUrl(taburl);
    if (ticketingSystemInfo.isCustomDomain == false) {
        if (existingCallback != null) {
            existingCallback(buildDomainObject(taburl, ticketingSystemInfo.ticketingSystem, false, tabid));
        }
    } else {
        let existingCustomDomain = await isCurrentDomainAlreadyAdded(taburl);
        if (existingCustomDomain != false) {
            existingCustomDomain.tabId = tabid;
            existingCallback(existingCustomDomain, true);
        }
        else {
            if (newCallback != null) {
                newCallback(tabid, taburl);
            }
        }
    }
}
const getCurrentUrl = function (currentUrl) {
    return new URL(currentUrl).origin + '/*';
}

const initiatePiaForKnownDomain = function (domainData) {
    //update the system type if required
    domainData.system = systemTypeElement.value;
    sendMessageToBackgroundService(MessageKeys.PIA_INITIATE_FOR_KNOWN_DOMAIN, domainData);
}

const initiatePiaForNewDomain = async function (tabid, taburl) {
    let selectedSystem = systemTypeElement.value;
    await addSelectedSystem(selectedSystem, getCurrentUrl(taburl), true, tabid);
}

addToPiaElement.onclick = async function () {
    let tab = await getCurrentActiveTab();
    let taburl = tab.url;
    let tabid = tab.id;
    await initiatePiaForNewDomain(tabid, taburl);
}

const addSelectedSystem = async function (system, serverAddress, isCustomDomain, tabid) {
    await sendMessageToBackgroundService(MessageKeys.PIA_ADD_CUSTOM_TICKETING_SYSTEM, buildDomainObject(serverAddress, system, isCustomDomain, tabid));
}

/**
 * @param {string} messageKey
 * @param {T} messageData
 * @returns {Promise<void>}
 */
const sendMessageToBackgroundService = async function (messageKey, messageData) {
    await chrome.runtime.sendMessage({ messageType: messageKey, data: messageData }, function (response) {
        window.close();
    });
}

const requestPiaWindowFromPopup = async (tabid) => {
    const message = {
        message: {
            messageType: MessageKeys.PIA_START_REQUEST_FROM_POPUP
        }
    };
    
    await chrome.tabs.sendMessage(tabid, message, async (response) => {
        if (response) {
            window.close();
        }
    });
}

window.addEventListener("load", async (event) => {
    let tab = await getCurrentActiveTab();
    let domainInfo = getTicketingSystemInfoFromUrl(tab.url);
    
    if (!domainInfo.isCustomDomain) {
        await requestPiaWindowFromPopup(tab.id);
    }
    else {
        let existingSystemInfo = await isCurrentDomainAlreadyAdded(tab.url);
        
        if (existingSystemInfo) {
            try {
                systemTypeElement.value = existingSystemInfo.system;
                
                // Check if the content script has already been injected on the page.
                // If it has, we can try to load the Pia window.
                const message = {
                    message: {
                        messageType: MessageKeys.CHECK_CONTENT_SCRIPT_LOADED
                    }
                };
                
                await chrome.tabs.sendMessage(tab.id, message, async (response) => {
                    if (chrome.runtime.lastError) {
                        // Message sending failed. It is expected as content script may not be available 
                        // during the first time.
                    }
                    else if (response) {
                        await requestPiaWindowFromPopup(tab.id);
                    }
                });
            }
            catch (error) {
                //failed setting system type in the dropdown
                //default will be set to HaloPSA
            }
        }
    }
});
